import React, { useState } from 'react';
import StyledButton from 'components/StyledButton';
import fillTemplate from 'es6-dynamic-template';
import Select from '@material-ui/core/Select';
import cn from 'classnames';
import MenuItem from '@material-ui/core/MenuItem';
import useStyles from './styles';
import { QCC_FW_LANGUAGES } from '../../../resources/strings/languages';

type Props = {
    strings: object,
    device: object,
    hasBetaVersion: boolean,
    betaVersion: string,
    onWhatsNew: () => {},
    onUpdateFirmware: () => {}
};

type FWLanguageProps = {
    device: object,
    fwLanguage: string,
    onLanguageSelect: string => {}
};

function RenderFWLanguage({
    device,
    fwLanguage,
    onLanguageSelect
}: FWLanguageProps) {
    const classes = useStyles();
    const language = fwLanguage || device.deviceLanguage;

    return (
        <Select
            className={cn(classes.language, classes.languageSelect)}
            value={language}
            onChange={event => onLanguageSelect(event.target.value)}
            MenuProps={{
                classes: {
                    paper: classes.languagePaper
                }
            }}
            disableUnderline
            inputProps={{
                classes: {
                    icon: classes.languageInputArrow,
                    root: classes.languageInput,
                    select: classes.languageSelect
                }
            }}
        >
            {Object.keys(QCC_FW_LANGUAGES)
                .filter(key =>
                    QCC_FW_LANGUAGES[key].models.includes(device.model)
                )
                .map(key => (
                    <MenuItem
                        className={cn(
                            classes.languageMenuItem,
                            key === language ? classes.selectedLanguageItem : ''
                        )}
                        key={key}
                        value={key}
                    >
                        <img alt={key} src={QCC_FW_LANGUAGES[key].icon} />{' '}
                        {QCC_FW_LANGUAGES[key].title}
                    </MenuItem>
                ))}
        </Select>
    );
}

function NewVersionAvailable({
    strings,
    device,
    hasBetaVersion,
    betaVersion,
    onWhatsNew,
    onUpdateFirmware
}: Props) {
    const classes = useStyles();
    const [fwLanguage, setFwLanguage] = useState(null);
    const isBetaInstalled = !!(
        betaVersion && betaVersion.toString() === device.fwVersion
    );
    const filteredLanguages = Object.keys(QCC_FW_LANGUAGES).filter(key =>
        QCC_FW_LANGUAGES[key].models.includes(device.model)
    );

    return (
        <div className={classes.newVersionAvailableContainer}>
            <div className={classes.currentVersion}>
                {fillTemplate(strings.device_firmware_version_is, {
                    version: device.fwVersion,
                    betaBuildNumber: `${
                        hasBetaVersion && isBetaInstalled && device.btmVersion
                            ? `(${device.btmVersion})`
                            : ''
                    }`
                })}
            </div>
            <div className={classes.newVersionAvailableTitle}>
                {strings.update_device_new_fw_available_title}
            </div>
            <div className={classes.newVersionAvailableSubContainer}>
                {onWhatsNew && (
                    <div className={classes.whatsNew} onClick={onWhatsNew}>
                        {strings.update_device_new_fw_available_release_link}
                    </div>
                )}
            </div>
            {device.deviceLanguage && filteredLanguages.length > 0 && (
                <div className={classes.languageContainer}>
                    <div className={classes.fwLanguageTitle}>
                        {strings.fw_language}
                    </div>
                    <RenderFWLanguage
                        device={device}
                        fwLanguage={fwLanguage || device.deviceLanguage}
                        onLanguageSelect={setFwLanguage}
                    />
                </div>
            )}
            <StyledButton
                onClick={() =>
                    onUpdateFirmware(fwLanguage || device.deviceLanguage)
                }
                btnSize="big"
            >
                {strings.update_device_new_fw_available_update_firmware}
            </StyledButton>
        </div>
    );
}

export default NewVersionAvailable;
